	//************************************************************
	//* Futaba S.Bus format (8-E-2/100Kbps)
	//*	S.Bus decoding algorithm borrowed in part from Arduino
	//*
	//* The protocol is 25 Bytes long and is sent every 14ms (analog mode) or 7ms (highspeed mode).
	//* One Byte = 1 startbit + 8 data bit + 1 parity bit + 2 stopbit (8E2), baudrate = 100,000 bit/s
	//*
	//* The highest bit is sent first. The logic is inverted.
	//*
	//* [startbyte] [data1] [data2] .... [data22] [flags][endbyte]
	//* 
	//* 0 startbyte = 11110000b (0xF0)
	//* 1-22 data = [ch1, 11bit][ch2, 11bit] .... [ch16, 11bit] (Values = 0 to 2047)
	//* 	channel 1 uses 8 bits from data1 and 3 bits from data2
	//* 	channel 2 uses last 5 bits from data2 and 6 bits from data3
	//* 	etc.
	//* 
	//* 23 flags = 
	//*	bit7 = ch17 = digital channel (0x80)
	//* 	bit6 = ch18 = digital channel (0x40)
	//* 	bit5 = Frame lost, equivalent red LED on receiver (0x20)
	//* 	bit4 = failsafe activated (0x10)
	//* 	bit3 = n/a
	//* 	bit2 = n/a
	//* 	bit1 = n/a
	//* 	bit0 = n/a
	//* 24 endbyte = 00000000b
	//*
	//************************************************************

IsrSBusGetData:
	in SregSaver, sreg
	push zh
	push zl
	ldz SBusByte0			;get start of data array

	;Read and check data
	load tt, ucsr0a			;read USART Control and Status Register A
	load treg, udr0			;read from USART buffer

	andi tt, 0x1C			;frame error, data overrun or parity error?
	brne rxerr

	;Start byte
	lds tt, SBusIndex
	tst tt
	brne rx2

	mov tt, treg			;check if start byte is valid
	cpi tt, 0x0F			;OBSERVE! Bit order is reversed when read from the USART buffer
	breq rx20
	rjmp rxerr			;invalid start byte found

rx2:	;End byte
	cpi tt, 24
	brne rx3

	tst treg
	breq rxok
	rjmp rxerr			;invalid end byte found

rxok:	;S.Bus frame is valid
	ser tt
	sts SBusFrameValid, tt		;set flag

	clr tt				;reset the timeout counter
	sts TimeOutCounter, tt

	sts SBusIndex, tt		;prepare for next S.Bus frame
	rjmp rxend

rx3:	;S.Bus flags
	cpi tt, 23
	brne rx4

	sts SBusFlags, treg
	rjmp rx20

rx4:	;Data byte
	andi tt, 0x1F			;prevent negative index values
	cpi tt, 12
	brge rx20

	lds zl, SBusByteAddressL
	lds zh, SBusByteAddressH
	st z+, treg

rx20:	;Increase the S.Bus index
	lds tt, SBusIndex
	inc tt
	sts SBusIndex, tt
	rjmp rxend

rxerr:	;S.Bus data error detected. Must re-synchronize the communication.
	lds tt, SBusFrameValid
	tst tt				;did we lose sync?
	breq rxerr1

	sts SBusDataErrors, tt		;yes, set error flag

rxerr1:	clr tt				;re-synchronize
	sts SBusFrameValid, tt
	sts SBusIndex, tt

rxend:	sts SBusByteAddressL, zl	;save array pointer before leaving
	sts SBusByteAddressH, zh
	pop zl
	pop zh
	out sreg, SregSaver
	reti


	;--- Read all input channel values ---

GetRxChannels:

	lds t, flagSBusFrameValid
	tst t				;is the S.Bus data frame valid?
	brne rx31

	rcall ClearInputChannels	;no, set all values to zero!
	ret


rx31:	;--- Roll ---

	;S.Bus channel 1 is 11 bit long and is stored in SBusByte0 and SBusByte1:
	;S.Bus data received:	76543210 -----A98
	;Variable name:		   xl	    xh

	cli				;get roll channel value
	lds yl, SBusIndex
	lds xl, SBusByte0
	lds xh, SBusByte1
	sei

	cpi yl, 1			;avoid using a mix of new and old data
	breq rx50

	andi xh, 0x07
	rcall AdjustSBusValue

	clr yh				;store in register
	b16store RxRoll
	b16store RxRollOld
	rjmp rx40

rx50:	b16mov RxRoll, RxRollOld	;use old value

	
rx40:	;--- Pitch

	;S.Bus channel 2 is 11 bit long and is stored in SBusByte1 and SBusByte2:
	;S.Bus data received:	43210--- --A98765
	;Variable name:		   xl	    xh

	cli				;get Pitch channel value
	lds yl, SBusIndex
	lds xl, SBusByte1
	lds xh, SBusByte2
	sei

	cpi yl, 2			;avoid using a mix of new and old data
	breq rx51

	ror xh
	ror xl
	ror xh
	ror xl
	ror xh
	ror xl
	andi xh, 0x07
	rcall AdjustSBusValue

	clr yh				;store in register
	b16store RxPitch
	b16store RxPitchOld
	rjmp rx41

rx51:	b16mov RxPitch, RxPitchOld	;use old value


rx41:	;--- Throttle ---

	;S.Bus channel 3 is 11 bit long and is stored in SBusByte2, SBusByte3 and SBusByte4:
	;S.Bus data received:	10------ 98765432 -------A
	;Variable name:		   yh	    xl	     xh

	cli				;get Throttle channel value
	lds yl, SBusIndex
	lds yh, SBusByte2
	lds xl, SBusByte3
	lds xh, SBusByte4
	sei

	cpi yl, 3			;avoid using a mix of new and old data
	breq rx52
	cpi yl, 4
	breq rx52

	rol yh
	rol xl
	rol xh
	rol yh
	rol xl
	rol xh
	andi xh, 0x07

	rvsetflagfalse flagThrottleZero

	ldz 400				;X = X - 400
	sub xl, zl
	sbc xh, zh
	rcall Add50Percent		;X = X * 1.5

	ldz 0				;X < 0 ?
	cp  xl, zl
	cpc xh, zh
	brge rx30

	ldx 0				;yes, set to zero
	rvsetflagtrue flagThrottleZero

rx30:	clr yh				;store in register
	b16store RxThrottle
	b16store RxThrottleOld
	rjmp rx42

rx52:	b16mov RxThrottle, RxThrottleOld;use old value



rx42:	;--- Yaw ---

	;S.Bus channel 4 is 11 bit long and is stored in SBusByte4 and SBusByte5:
	;S.Bus data received:	6543210- ----A987
	;Variable name:    	   xl       xh

	cli				;get Yaw channel value
	lds yl, SBusIndex
	lds xl, SBusByte4
	lds xh, SBusByte5
	sei

	cpi yl, 5			;avoid using a mix of new and old data
	breq rx53

	ror xh
	ror xl
	andi xh, 0x07
	rcall AdjustSBusValue

	clr yh
	b16store RxYaw
	b16store RxYawOld
	rjmp rx43

rx53:	b16mov RxYaw, RxYawOld		;use old value



rx43:	;--- AUX ---

	;S.Bus channel 5 is 11 bit long and is stored in SBusByte5 and SBusByte6:
	;S.Bus data received:	3210---- -A987654
	;Variable name:    	   xl       xh

	cli				;get Aux channel value
	lds yl, SBusIndex
	lds xl, SBusByte5
	lds xh, SBusByte6
	sei

	cpi yl, 6			;avoid using a mix of new and old data
	breq rx54

	ror xh
	ror xl
	ror xh
	ror xl
	ror xh
	ror xl
	ror xh
	ror xl
	andi xh, 0x07
	rcall AdjustSBusValue

	clr t				;clear flags
	sts flagSlStickMixing, t
	sts flagSlOn, t

	ldz 625
	cp  xl, zl
	cpc xh, zh
	brge rx33

	ldz -625
	cp  xl, zl
	cpc xh, zh
	brge rx34
	rjmp rx35			;AUX switch in OFF position

rx33:	rvsetflagtrue flagSlOn		;AUX switch in ON position
	rjmp rx35

rx34:	rvsetflagtrue flagSlStickMixing	;AUX switch in middle position

rx35:	clr yh				;store in register
	b16store RxAux
	b16store RxAuxOld
	rjmp rx44

rx54:	b16mov RxAux, RxAuxOld		;use old value


rx44:	;--- AUX2 ---

	;S.Bus channel 6 is 11 bit long and is stored in SBusByte6, SBusByte7 and SBusByte8:
	;S.Bus data received:	0------- 87654321 ------A9
	;Variable name:    	   yh       xl       xh

	cli				;get Aux2 channel value
	lds yl, SBusIndex
	lds yh, SBusByte6
	lds xl, SBusByte7
	lds xh, SBusByte8
	sei

	cpi yl, 7			;avoid using a mix of new and old data
	breq rx55
	cpi yl, 8
	breq rx55

	rol yh
	rol xl
	rol xh
	andi xh, 0x07
	rcall AdjustSBusValue

	clr yh				;store in register
	b16store RxAux2
	b16store RxAux2Old
	rjmp rx45

rx55:	b16mov RxAux2, RxAux2Old	;use old value


rx45:	;--- AUX3 ---

	;S.Bus channel 7 is 11 bit long and is stored in SBusByte8 and SBusByte9:
	;S.Bus data received:	543210-- ---A9876
	;Variable name:    	   xl       xh

	cli				;get Aux2 channel value
	lds yl, SBusIndex
	lds xl, SBusByte8
	lds xh, SBusByte9
	sei

	cpi yl, 9			;avoid using a mix of new and old data
	breq rx56

	ror xh
	ror xl
	ror xh
	ror xl
	andi xh, 0x07
	rcall AdjustSBusValue

	clr yh				;store in register
	b16store RxAux3
	b16store RxAux3Old
	rjmp rx46

rx56:	b16mov RxAux3, RxAux3Old	;use old value


rx46:	;--- AUX4 ---

	;S.Bus channel 8 is 11 bit long and is stored in SBusByte9 and SBusByte10:
	;S.Bus data received:	210----- A9876543
	;Variable name:    	   xl       xh

	cli
	lds yl, SBusIndex
	lds xl, SBusByte9
	lds xh, SbusByte10
	sei

	cpi yl, 10			;avoid using a mix of new and old data
	breq rx57

	ldi t, 5
rx58:	lsr xh
	ror xl
	dec t
	brne rx58

	rcall AdjustSBusValue

	clr yh
	b16store RxAux4
	b16store RxAux4Old
	rjmp rx47

rx57:	b16mov RxAux4, RxAux4Old	;use old value


	;--- Use a [-100, 100] range for Aux2, Aux3 and Aux4 ---

rx47:	b16ldi Temp, 0.1
	b16mul RxAux2, RxAux2, Temp
	b16mul RxAux3, RxAux3, Temp
	b16mul RxAux4, RxAux4, Temp
	ret



	;---- Clear all input values ---

ClearInputChannels:

	b16clr RxRoll
	b16set RxPitch
	b16set RxThrottle
	b16set RxYaw
	b16set RxAux
	b16set RxAux2
	b16set RxAux3
	b16set RxAux4
	rvsetflagtrue flagThrottleZero
	rvsetflagfalse flagSlOn
	ret



	;---- Adapt S.Bus value to KK value ---

AdjustSBusValue:	;Subtract Futaba S.Bus offset (1024) and multiply by 1.5

	ldz 1024	;X = X - 1024
	sub xl, zl
	sbc xh, zh

Add50Percent:

	mov zl, xl	;Z = X / 2
	mov zh, xh
	asr zh
	ror zl

	add xl, zl	;X = X + Z
	adc xh, zh
	ret

